<?php

class QueryBuilder
{
    private PDO $pdo;
    private string $table = '';
    private array $fields = ['*'];
    private array $wheres = [];
    private array $bindings = [];
    private array $joins = [];
    private ?string $order = null;
    private ?int $limit = null;
    private string $queryType = 'select';
    private array $insertData = [];
    private array $updateData = [];

    public function __construct(PDO $pdo)
    {
        $this->pdo = $pdo;
    }

    /** ------------------------------ **/
    /**        USTAWIENIA PODSTAWOWE   **/
    /** ------------------------------ **/

    public function table(string $table): self
    {
        $this->reset();
        $this->table = $table;
        return $this;
    }

    public function select(array $fields = ['*']): self
    {
        $this->queryType = 'select';
        $this->fields = $fields;
        return $this;
    }

    public function insert(array $data): self
    {
        $this->queryType = 'insert';
        $this->insertData = $data;
        return $this;
    }

    public function update(array $data): self
    {
        $this->queryType = 'update';
        $this->updateData = $data;
        return $this;
    }

    public function delete(): self
    {
        $this->queryType = 'delete';
        return $this;
    }

    /** ------------------------------ **/
    /**            WARUNKI              **/
    /** ------------------------------ **/

    public function where(string $column, string $operator, $value): self
    {
        $placeholder = ':' . str_replace('.', '_', $column) . count($this->bindings);
        $this->wheres[] = "$column $operator $placeholder";
        $this->bindings[$placeholder] = $value;
        return $this;
    }

    public function orderBy(string $column, string $direction = 'ASC'): self
    {
        $this->order = "$column $direction";
        return $this;
    }

    public function limit(int $limit): self
    {
        $this->limit = $limit;
        return $this;
    }

    /** ------------------------------ **/
    /**             JOINY               **/
    /** ------------------------------ **/

    public function join(string $table, string $first, string $operator, string $second): self
    {
        $this->joins[] = "INNER JOIN $table ON $first $operator $second";
        return $this;
    }

    public function leftJoin(string $table, string $first, string $operator, string $second): self
    {
        $this->joins[] = "LEFT JOIN $table ON $first $operator $second";
        return $this;
    }

    /** ------------------------------ **/
    /**          BUDOWANIE SQL          **/
    /** ------------------------------ **/

    private function buildSelect(): string
    {
        $sql = "SELECT " . implode(', ', $this->fields) . " FROM {$this->table}";

        if (!empty($this->joins)) {
            $sql .= ' ' . implode(' ', $this->joins);
        }

        if (!empty($this->wheres)) {
            $sql .= " WHERE " . implode(' AND ', $this->wheres);
        }

        if ($this->order) {
            $sql .= " ORDER BY {$this->order}";
        }

        if ($this->limit) {
            $sql .= " LIMIT {$this->limit}";
        }

        return $sql;
    }

    private function buildInsert(): string
    {
        $columns = implode(', ', array_keys($this->insertData));
        $placeholders = ':' . implode(', :', array_keys($this->insertData));
        foreach ($this->insertData as $k => $v) {
            $this->bindings[":$k"] = $v;
        }
        return "INSERT INTO {$this->table} ($columns) VALUES ($placeholders)";
    }

    private function buildUpdate(): string
    {
        $set = [];
        foreach ($this->updateData as $k => $v) {
            $placeholder = ":upd_$k";
            $set[] = "$k = $placeholder";
            $this->bindings[$placeholder] = $v;
        }

        $sql = "UPDATE {$this->table} SET " . implode(', ', $set);

        if (!empty($this->wheres)) {
            $sql .= " WHERE " . implode(' AND ', $this->wheres);
        }

        return $sql;
    }

    private function buildDelete(): string
    {
        $sql = "DELETE FROM {$this->table}";
        if (!empty($this->wheres)) {
            $sql .= " WHERE " . implode(' AND ', $this->wheres);
        }
        return $sql;
    }

    /** ------------------------------ **/
    /**         WYKONYWANIE SQL         **/
    /** ------------------------------ **/

    public function get()
    {
        $sql = match ($this->queryType) {
            'select' => $this->buildSelect(),
            'insert' => $this->buildInsert(),
            'update' => $this->buildUpdate(),
            'delete' => $this->buildDelete(),
        };

        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($this->bindings);

        return match ($this->queryType) {
            'select' => $stmt->fetchAll(PDO::FETCH_ASSOC),
            default => $stmt->rowCount(),
        };
    }

    /** ------------------------------ **/
    /**           TRANSAKCJE            **/
    /** ------------------------------ **/

    public function beginTransaction(): void
    {
        $this->pdo->beginTransaction();
    }

    public function commit(): void
    {
        $this->pdo->commit();
    }

    public function rollBack(): void
    {
        $this->pdo->rollBack();
    }

    /** ------------------------------ **/
    /**         RESET STANU BUILDERA    **/
    /** ------------------------------ **/

    private function reset(): void
    {
        $this->fields = ['*'];
        $this->wheres = [];
        $this->bindings = [];
        $this->joins = [];
        $this->order = null;
        $this->limit = null;
        $this->insertData = [];
        $this->updateData = [];
        $this->queryType = 'select';
    }
}

